/* $Id: srcdata.c,v 1.61 1998/12/16 20:43:36 ericb Exp $ */
/* Copyright (C) 1994 - 1998, Hewlett-Packard Company, all rights reserved. */
/* Created by Don Mathiesen, derived from getdata.c */

/* contains arb source data transfer functions */

#include "sema.h"

/* #define PRINTBDPARMS to print source board cal factors */
/* #define DPRINT */
/* #define DPRINT3 */

#define BDPARM_SIZE	60

#define SRC_CHECK(x)			\
    {					\
	error_src = (x);			\
	if ( error_src < 0 ) return error_src;	\
    }

/*
 *********************************************************************
 Do the data transfer by writing directly to VXI memory.  The
 substrate 96002 doesn't take any part in the transfer, but will be
 checking and unloading the buffer in the background, if the source is
 in E1432_SOURCE_MODE_ARB mode.

 This is a general block transfer from host to 96002 memory and is used
 elsewhere.
 *********************************************************************
 */
SHORTSIZ16
i1432_xfer_writedata(E1432_MODULE_LIST_NODE *mn, LONGSIZ32 *ldata,
		     LONGSIZ32 lsubstbuf, LONGSIZ32 length)
{
    /*
      mn = module node
      *ldata = pointer to host address of 32bit data
      lsubstbuf = a pointer to substrate RAM byte address
      length = # of 32bit data words
    */

    SHORTSIZ16 error;

#ifdef DPRINT3
    (void)printf("i1432_xfer_writedata: mn = 0x%x, lsubstbuf = %x, length = %d\n", mn, lsubstbuf, length);
#endif   

#if SWAP
    if (!mn->d32)
    {
	/* The data is 32-bit data.  If the data will get transferred
	   (and byte swapped) in 16-bit chunks, we need to swap the
	   data now to fix it. */
	error = i1432_two_four_swap((char *) ldata, length * 4);
	if (error)
	    return error;
    }
#endif

    /* (void) printf("(sram) i1432_xfer_block(%d, 0x%p, 0x%p, %lu, 0, 0)\n",
       mn - i1432_mod_list, ldata, lsubstbuf, length); */
    error = i1432_xfer_block(mn, (unsigned long *) ldata,
			     (unsigned long *) lsubstbuf,
			     length, 0, 0);
    if (error)
	return error;

    return 0;
}

static SHORTSIZ16
i1432_xfer_writedata_to_dram(E1432_MODULE_LIST_NODE *mn, LONGSIZ32 *ldata,
			     LONGSIZ32 lsubstbuf, LONGSIZ32 length)
{
    SHORTSIZ16 error, error2;
    unsigned long *start;
    long    page, window, offset, xfer;

#if SWAP
    if (!mn->d32)
    {
	/* The data is 32-bit data.  If the data will get transferred
	   (and byte swapped) in 16-bit chunks, we need to swap the
	   data now to fix it. */
	error = i1432_two_four_swap((char *) ldata, length * 4);
	if (error)
	    goto cleanup;
    }
#endif

    /* Get page map, window size, and offset into window */
    page = E1432_DRAM_PAGE;
    if (mn->a24_256k)
    {
	window = E1432_WINDOW_BASE_256K;
	page <<= 2;
    }
    else
	window = E1432_WINDOW_BASE_1M;
    page += lsubstbuf / window;
    offset = lsubstbuf % window;
    /* Convert to 32-bit words */
    offset >>= 2;
    window >>= 2;

    /* Get start xfer address for window */

    /* When using a24_256k, or when using VTL, we map in only the
       movable window, so the address we want is just the base address
       that was imapped. */
    start = 0;
#ifndef	HAVE_VTL
    if (!mn->a24_256k)
	/* When not using a24_256k and not using VTL, we map all of
	   the module's A24 space, so we want the address of the
	   movable window. */
	start = (unsigned long *) window;
#endif

    while (length > 0)
    {
	/* Set up page map register */
	/* (void) printf("page map 0x%lx\n", page); */
	error = i1432_direct_write_register(mn, E1432_PAGE_MAP_REG,
					    (SHORTSIZ16) page);
	if (error)
	    goto cleanup;

	/* How many words to transfer? */
	xfer = length;
	if (xfer > window - offset)
	    xfer = window - offset;

	/* Transfer data */
	/* (void) printf("(dram) i1432_xfer_block(%d, 0x%p, 0x%p, %lu, 0, 0)\n",
	   mn - i1432_mod_list, ldata, start + offset, xfer); */
	/* Avoid pointer math because 16-bit windows does it wrong */
	error = i1432_xfer_block(mn, (unsigned long *) ldata,
				 (unsigned long *)
				 ((unsigned long) start +
				  offset * sizeof (unsigned long)),
				 xfer, 0, 0);
	if (error)
	    goto cleanup;

	/* Prepare for next loop */
	length -= xfer;
	/* Avoid pointer math because 16-bit windows does it wrong */
	ldata = (LONGSIZ32 *) ((unsigned long) ldata +
			       xfer * sizeof (unsigned long));
	offset = 0;
	page++;
    }

 cleanup:
    /* Clean up page map register */
    error2 = i1432_direct_write_register(mn, E1432_PAGE_MAP_REG,
					 E1432_BASE_PAGE);
    if (error2)
	error = error2;

    return error;
}

/*
 *********************************************************************
 Do the data transfer by writing directly to VXI memory.  The
 substrate 96002 doesn't take any part in the transfer, but will be
 checking and unloading the buffer in the background, if the source is
 in E1432_SOURCE_MODE_ARB mode.
 *********************************************************************
 */
static SHORTSIZ16
get_xfer_regs(E1432ID hw, SHORTSIZ16 chanID, LONGSIZ32 *xfrwdsreg,
	      LONGSIZ32 *statusregA, LONGSIZ32 *statusregB,
	      LONGSIZ32 *cmdreg, LONGSIZ32 *bufreg, int *dram)
{
    /*
      hw = hardware id
      chanID = chanID
      xfrwdsreg = pointer to transfer buffer word count reg
      statusregA = pointer to source bufferA status reg
      statusregB = pointer to source bufferB status reg
      cmdreg = pointer to transfer command reg
      bufreg = pointer to transfer buffer
    */
    E1432_MODULE_LIST_NODE *mn;
    SHORTSIZ16 error_src;
    long   limit;
    int    sca, index;

    TRACE_PRINTF(2, ("get_xfer_regs(0x%p, %d)\n", hw, chanID));

    sca = i1432_get_chan_sca(hw, chanID);
    index = i1432_get_chan_index(hw, chanID);

    SRC_CHECK(i1432_get_module_from_chan(hw, chanID, &mn));

    /*
      map to odd and even since only 2 source channels per clarion and all
      other lower numbered scas have an  even # of channels
      index starts at 0 for channel 1
      only one source channel on sca 4, the source daughter board
    */
    if ((index & 0x1) == 1)
    {
	/* channel 2 of 2 */
	switch (sca)
	{
	case 0:
	    *statusregA = (LONGSIZ32)E1432_SRC0_2_XFRA_STATUS_REG;
	    *statusregB = (LONGSIZ32)E1432_SRC0_2_XFRB_STATUS_REG;
	    *xfrwdsreg = (LONGSIZ32)E1432_SRC0_2_XFRBUFFER_WORDCOUNT_REG;
	    *cmdreg = (LONGSIZ32)E1432_SRC0_2_XFRCMD_REG;
	    SRC_CHECK(i1432_direct_read32_register(mn,
						   E1432_SRC0_2_XFRBUFPTR_REG,
						   bufreg));
	    /* Was: *bufreg = (LONGSIZ32)E1432_SRC0_2_XFRBUFFER_START_REG; */
	    break;
	case 2:
	    *statusregA = (LONGSIZ32)E1432_SRC2_2_XFRA_STATUS_REG;
	    *statusregB = (LONGSIZ32)E1432_SRC2_2_XFRB_STATUS_REG;
	    *xfrwdsreg = (LONGSIZ32)E1432_SRC2_2_XFRBUFFER_WORDCOUNT_REG;
	    *cmdreg = (LONGSIZ32)E1432_SRC2_2_XFRCMD_REG;
	    SRC_CHECK(i1432_direct_read32_register(mn,
						   E1432_SRC2_2_XFRBUFPTR_REG,
						   bufreg));
	    /* Was: *bufreg = (LONGSIZ32)E1432_SRC2_2_XFRBUFFER_START_REG; */
	    break;
	default:
	    return i1432_print_error(ERR1432_INTERNAL_LIBRARY_ERROR);
	}
    }
    else
    {
	/* channel 1 of 2 */
	switch (sca)
	{
	case 0:
	    *statusregA = (LONGSIZ32)E1432_SRC0_XFRA_STATUS_REG;
	    *statusregB = (LONGSIZ32)E1432_SRC0_XFRB_STATUS_REG;
	    *xfrwdsreg = (LONGSIZ32)E1432_SRC0_XFRBUFFER_WORDCOUNT_REG;
	    *cmdreg = (LONGSIZ32)E1432_SRC0_XFRCMD_REG;
	    SRC_CHECK(i1432_direct_read32_register(mn,
						   E1432_SRC0_XFRBUFPTR_REG,
						   bufreg));
	    /* Was: *bufreg = (LONGSIZ32)E1432_SRC0_XFRBUFFER_START_REG; */
	    break;
	case 2:
	    *statusregA = (LONGSIZ32)E1432_SRC2_XFRA_STATUS_REG;
	    *statusregB = (LONGSIZ32)E1432_SRC2_XFRB_STATUS_REG;
	    *xfrwdsreg = (LONGSIZ32)E1432_SRC2_XFRBUFFER_WORDCOUNT_REG;
	    *cmdreg = (LONGSIZ32)E1432_SRC2_XFRCMD_REG;
	    SRC_CHECK(i1432_direct_read32_register(mn,
						   E1432_SRC2_XFRBUFPTR_REG,
						   bufreg));
	    /* Was: *bufreg = (LONGSIZ32)E1432_SRC2_XFRBUFFER_START_REG; */
	    break;
	case 4:
	    *statusregA = (LONGSIZ32)E1432_SRC4_XFRA_STATUS_REG;
	    *statusregB = (LONGSIZ32)E1432_SRC4_XFRB_STATUS_REG;
	    *xfrwdsreg = (LONGSIZ32)E1432_SRC4_XFRBUFFER_WORDCOUNT_REG;
	    *cmdreg = (LONGSIZ32)E1432_SRC4_XFRCMD_REG;
	    SRC_CHECK(i1432_direct_read32_register(mn,
						   E1432_SRC4_XFRBUFPTR_REG,
						   bufreg));
	    /* Was: *bufreg = (LONGSIZ32)E1432_SRC4_XFRBUFFER_START_REG; */
	    break;
	default:
	    return i1432_print_error(ERR1432_INTERNAL_LIBRARY_ERROR);
	}
    }

    /* Semi-kludge.  Assume that "small" addresses are directly
       available in B-Bus sram, while "large" addresses are in
       A-Bus DRAM. */
    if (*bufreg >= E1432_DRAM_START)
    {
	*dram = 1;
	*bufreg -= E1432_DRAM_START; /* Subtract out DRAM base */
	limit = E1432_DRAM_SIZE_MAX;
    }
    else
    {
	*dram = 0;
	limit = E1432_WINDOW_BASE_256K;
    }
    if (*bufreg > limit)
	/* Something wrong somewhere, probably wrong sema.bin */
	return i1432_la_print_error(mn->la, ERR1432_SEMA_BIN_MISMATCH);

#ifdef DPRINT
    (void) printf("get_xfer_regs: chanID %d, sca %d, index %d, "
		  "*xfrwdsreg 0x%x, *bufreg 0x%x, *dram %d\n",
		  chanID, sca, index, *xfrwdsreg, *bufreg, *dram);
#endif
    return 0;
}

/*
 *********************************************************************
 Return a 1 (TRUE) if there is another active channel in this source
 sca with  an empty transfer buffer.  Otherwise returns a zero (FALSE).
 This check is done to avoid hanging on the first channel, during an
 arb data xfer with wait, when 2 channels of data are needed for
 interleaving (to transfer data from the substrate to the source board
 dsp).
 *********************************************************************
 */
static SHORTSIZ16
xfer_check_nowait(E1432ID hw, SHORTSIZ16 chanID)
{
    SHORTSIZ16 error_src;
    SHORTSIZ16 otherchanID, otherstate;
    LONGSIZ32 otherxfrbuf_wds, othersrcbuf_state_AB, othersrcbuf_state_A;
    LONGSIZ32 othersrcbuf_state_B;
 
    /* if (there is more than one channel on this sca) AND
	   { if (the other channel is active) AND
		{ if (the [from get_xfer_reg for the other chan_id] *xfrwdsreg == 0)
		     {
			nowait = 1;
		      }
		}
	   } 
    */

    otherchanID = i1432_get_chan_other(hw, chanID);
    if (otherchanID != 0)
    {
	SRC_CHECK(e1432_get_active(hw, otherchanID, &otherstate));
	if (otherstate == E1432_CHANNEL_ON)
	{
	    SRC_CHECK(e1432_get_src_arbstates(hw, otherchanID,
					      &otherxfrbuf_wds,
					      &othersrcbuf_state_AB,
					      &othersrcbuf_state_A,
					      &othersrcbuf_state_B));
	    if (otherxfrbuf_wds == 0)
		return 1;
	}
    }

    return 0;
}

/****************************************************************************
 Writes a block of ARB source data of length numwords, pointed to by dataPt,
 to the first available substrate buffer.  

 e1432_check_src_arbrdy() should be called prior to calling this
 function in order to determine if more data can be accepted.

 mode is one of:
   E1432_SRC_DATA_MODE_AB - transfer to A or B buffer, whichever is
               available (preference to A if both are available).
   E1432_SRC_DATA_MODE_A - transfer to A buffer if available.
   E1432_SRC_DATA_MODE_B - transfer to B buffer if available.

   The following modes are same as above but wait for transfer to
   complete before returning.

   E1432_SRC_DATA_MODE_WAITAB
   E1432_SRC_DATA_MODE_WAITA
   E1432_SRC_DATA_MODE_WAITB

The transfer will be a block transfer if the source is off, or a "clocked"
transfer if the source is on.  This is taken care of by clari.c in the
substrate, code.

 Returns 0 if OK.
 Returns 1 if failed.
 Returns <0 an error number.

 ****************************************************************************/

SHORTSIZ16 EXPORT
e1432_write_srcbuffer_data(E1432ID hw, SHORTSIZ16 chanID, LONGSIZ32 * dataPtr,
			   LONGSIZ32 numwords, SHORTSIZ16 mode)
{
    E1432_MODULE_LIST_NODE *mn;
    SHORTSIZ16 error_src;
    LONGSIZ32 xfr_statusA, xfr_statusB, xfrcmd;
    LONGSIZ32 statusregA, statusregB, xfrwdsreg, cmdreg, bufreg;
    LONGSIZ32 xfrwds, xfrnumwords, srcbuffer_size;
    LONGSIZ32 *data_from_ptr, *data_to_ptr;
    FLOATSIZ64 startTime, currentTime, timeout_xfrwait = 10;
    int     i, dram;

    /* Used only for debug: */
#ifdef DPRINT3
    LONGSIZ32 xfrcmd2, xfrwds2, xfrwds3, xfrwds4;

    (void)printf("e1432_write_srcbuffer_data(hw (0x%p), chanID (%x), dataPtr (0x%p), numwords (%d),  mode (%d))\n", hw, chanID, dataPtr, numwords, mode);
#endif

    TRACE_PRINTF(0, ("e1432_write_srcbuffer_data(0x%p, %d, 0x%p, %d, %d)\n",
                     hw, chanID, dataPtr, numwords, mode));

    /* check if real channel */
    SRC_CHECK(i1432_checkID(hw, chanID));

    if (chanID < 0)
	return i1432_print_error(ERR1432_EXPECT_CHANNEL);

    SRC_CHECK(i1432_get_module_from_chan(hw, chanID, &mn));

    /* get substrate buffer destination or return with error if none
       available */
    SRC_CHECK(get_xfer_regs(hw, chanID, &xfrwdsreg, &statusregA,
			    &statusregB, &cmdreg, &bufreg, &dram));

    /* Is transfer buffer available ? */
    SRC_CHECK(i1432_direct_read32_register(mn, statusregA, &xfr_statusA));
    SRC_CHECK(i1432_direct_read32_register(mn, statusregB, &xfr_statusB));
    SRC_CHECK(i1432_direct_read32_register(mn, xfrwdsreg, &xfrwds));
#ifdef DPRINT3
    (void)printf("e1432_write_srcbuffer_data: xfrwdsreg %x, xfrwds %d\n",
		 xfrwdsreg, xfrwds);
#endif
    if (xfrwds != 0 && xfrwds != -1)
	return i1432_la_print_error(mn->la, ERR1432_SRC0_XFRBUF_NRDY);

    /* check to see if the data will fit in the buffer */
    if (dram)
    {
	SRC_CHECK(e1432_get_srcbuffer_size(hw, chanID, &srcbuffer_size));
	if (numwords > srcbuffer_size)
	    return i1432_print_error(ERR1432_SRC0_XFRDATA_TOO_LONG);
    }
    else
    {
	if (numwords > E1432_SRC_DATA_NUMWORDS_MAX)
	    return i1432_print_error(ERR1432_SRC0_XFRDATA_TOO_LONG);
    }

    xfrnumwords = numwords;

    switch (mode)
    {
    case E1432_SRC_DATA_MODE_RAWDATA:
	xfrcmd = E1432_SRC_XFRCMD_DATA;
	break;

    case E1432_SRC_DATA_MODE_RAWNACKCMD:
	xfrcmd = E1432_SRC_XFRCMD_NACKCMD;
	break;

    case E1432_SRC_DATA_MODE_RAWCMD:
	xfrcmd = E1432_SRC_XFRCMD_CMD;
	break;

    case E1432_SRC_DATA_MODE_RAWREAD:
	xfrcmd = E1432_SRC_XFRCMD_READ;
	xfrnumwords = -1-numwords;
	break;

    case E1432_SRC_DATA_MODE_READNCLRSTATUS:
	xfrcmd = E1432_SRC_XFRCMD_READNCLRSTATUS;
	xfrnumwords = -2;
	numwords = 1;
	break;

    case E1432_SRC_DATA_MODE_READSTATUS:
	xfrcmd = E1432_SRC_XFRCMD_READSTATUS;
	xfrnumwords = -2;
	numwords = 1;
	break;

    case E1432_SRC_DATA_MODE_READID:
	xfrcmd = E1432_SRC_XFRCMD_READID;
	xfrnumwords = -2;
	numwords = 1;
	break;

    case E1432_SRC_DATA_MODE_RESETA:
	xfrcmd = E1432_SRC_XFRCMD_RESETA;
	break;

    case E1432_SRC_DATA_MODE_RESETB:
	xfrcmd = E1432_SRC_XFRCMD_RESETB;
	break;

    case E1432_SRC_DATA_MODE_A:
    case E1432_SRC_DATA_MODE_WAITA:
	/* Abort if A is full */
	if ((xfr_statusA & 0xff) == E1432_SRCBUF_FULL)
	    return i1432_la_print_error(mn->la, ERR1432_SRC0_XFRBUF_NRDY);
	if ((xfr_statusA & 0xff) == E1432_SRCBUF_AVAIL)
	    xfrcmd = E1432_SRC_XFRCMD_A;
	else
	    xfrcmd = E1432_SRC_XFRCMD_CONTA;
	break;

    case E1432_SRC_DATA_MODE_B:
    case E1432_SRC_DATA_MODE_WAITB:
	/* Abort if B is full */
	if ((xfr_statusB & 0xff) == E1432_SRCBUF_FULL)
	    return i1432_la_print_error(mn->la, ERR1432_SRC0_XFRBUF_NRDY);
	if ((xfr_statusB & 0xff) == E1432_SRCBUF_AVAIL)
	    xfrcmd = E1432_SRC_XFRCMD_B;
	else
	    xfrcmd = E1432_SRC_XFRCMD_CONTB;
	break;
    case E1432_SRC_DATA_MODE_AB:
    case E1432_SRC_DATA_MODE_WAITAB:
	/* Abort if A and B are full */
	if ((xfr_statusA & 0xff) == E1432_SRCBUF_FULL &&
	    (xfr_statusB & 0xff) == E1432_SRCBUF_FULL)
	    return i1432_la_print_error(mn->la, ERR1432_SRC0_XFRBUF_NRDY);
	xfrcmd = E1432_SRC_XFRCMD_AB;
	break;

    default:
	return i1432_print_error(ERR1432_ILLEGAL_SRC_DATA_MODE);
    }
    
    /* copy data */
	
    /*
      SRC_CHECK(i1432_direct_read32_register(mn, (LONGSIZ32) xfrwdsreg,
      &xfrwds4));
      SRC_CHECK(i1432_direct_read32_register(mn, (LONGSIZ32) cmdreg,
      &xfrcmd2));
    */

    if (xfrnumwords > 0)	/* write data */
    {
	if (dram)
	    error_src = i1432_xfer_writedata_to_dram(mn, dataPtr,
						     bufreg, xfrnumwords);
	else
	    error_src = i1432_xfer_writedata(mn, dataPtr,
					     bufreg, xfrnumwords);
	if (error_src)
	    return error_src;
    }

#ifdef DPRINT3
    (void)printf("e1432_write_srcbuffer_data: error_src %d, xfrnumwords %d\n",error_src,xfrnumwords);
#endif
 
    /* indicate buffer is full by setting WORDCOUNT_REG or if <0 ready for read */
    /* effectively updating e1432_get_src_arbstates() */
#ifdef DPRINT3
    (void) printf("e1432_write_srcbuffer_data write: xfrwdsreg 0x%x, xfrnumwords %d\n",
		  xfrwdsreg, xfrnumwords);
#endif
    SRC_CHECK (i1432_direct_write32_register(mn, (LONGSIZ32)xfrwdsreg, (LONGSIZ32) xfrnumwords));
    /*
      SRC_CHECK (i1432_direct_read32_register(mn,(LONGSIZ32) xfrwdsreg, &xfrwds3 ));
    */

    if (xfrnumwords != 0)	/* write cmd */
    {
	/* set xfrcmd */
#ifdef DPRINT3
	(void) printf("e1432_write_srcbuffer_data write: cmdreg 0x%x, xfrcmd 0x%x\n",
		      cmdreg, xfrcmd);
#endif
	SRC_CHECK(i1432_direct_write32_register(mn, (LONGSIZ32) cmdreg,
						(LONGSIZ32) xfrcmd));
    }

    /* wait if needed */
    switch (mode)
    {
    case E1432_SRC_DATA_MODE_RAWCMD:
	/* is cmd done ? */
	/* xfrwds == -1 */
	/*
	  printf("bufer_xfr_RAWCMD: xfrcmdb4 (%x), xfrwdsb4 (%d) mode (%x), xfrwds (%d)\n",xfrcmd2,xfrwds4,mode,xfrwds3);
	*/
	i1432_get_time(&startTime);
	do
	{
	    i1432_get_time(&currentTime);
	    SRC_CHECK(i1432_direct_read32_register(mn, 
						   (LONGSIZ32) xfrwdsreg,
						   &xfrwds));

#ifdef DPRINT
	    SRC_CHECK(i1432_direct_read32_register(mn,
						   (LONGSIZ32) cmdreg,
						   &xfrcmd2));
	    printf("xfr_RAWCMD: mode 0x%x, xfrwds %d, xfrcmdreg 0x%x\n",
		   mode, xfrwds, xfrcmd2);
#endif
	    if (xfrwds != -1 && currentTime - startTime >= timeout_xfrwait)
		return i1432_la_print_error(mn->la, ERR1432_ACK_TIMEOUT);
	}
	while (xfrwds != -1);

	/* indicate buffer is empty by setting WORDCOUNT_REG to 0 */
	SRC_CHECK(i1432_direct_write32_register(mn, (LONGSIZ32) xfrwdsreg, 0));
#ifdef DPRINT
	SRC_CHECK (i1432_direct_read32_register(mn, 
						(LONGSIZ32) xfrwdsreg, &xfrwds2 ));
	SRC_CHECK (i1432_direct_read32_register(mn, 
						(LONGSIZ32) cmdreg, &xfrcmd2 ));
	printf("xfr_RAWCMD: mode (%x), xfrwds (%d), xfrcmdreg (%x)\n\n",mode,xfrwds2,xfrcmd2);
#endif
	break;

    case E1432_SRC_DATA_MODE_RAWNACKCMD:
	/* is transfer done ? */
	/* xfrwds == 0 */
	i1432_get_time(&startTime);
	do
	{
 	    i1432_get_time(&currentTime);
	    SRC_CHECK(i1432_direct_read32_register(mn, 
						   (LONGSIZ32) xfrwdsreg,
						   &xfrwds));
#ifdef DPRINT
	    printf("xfr_rawnackcmd: mode (%d), xfrwds (%d)\n",mode,xfrwds);
#endif
	    if (xfrwds > 0 && currentTime - startTime >= timeout_xfrwait)
		return i1432_la_print_error(mn->la, ERR1432_ACK_TIMEOUT);
	}
	while (xfrwds > 0);

	SRC_CHECK (i1432_direct_write32_register(mn,
						 (LONGSIZ32) cmdreg,
						 E1432_SRC_XFRCMD_IDLE));

	/* indicate buffer is empty by setting WORDCOUNT_REG to 0 */
	SRC_CHECK(i1432_direct_write32_register(mn, (LONGSIZ32) xfrwdsreg, 0));
	break;

    case E1432_SRC_DATA_MODE_RAWDATA:
	/* is transfer done ? */
	/* xfrwds == 0 */
	i1432_get_time(&startTime);
	do
	{
 	    i1432_get_time(&currentTime);
	    SRC_CHECK(i1432_direct_read32_register(mn, 
						   (LONGSIZ32) xfrwdsreg,
						   &xfrwds));
#ifdef DPRINT
	    printf("xfr_rawdata: mode (%d), xfrwds (%d)\n",mode,xfrwds);
#endif
	    if (xfrwds > 0 && currentTime - startTime >= timeout_xfrwait)
		return i1432_la_print_error(mn->la, ERR1432_ACK_TIMEOUT);
	}
	while (xfrwds > 0);

	SRC_CHECK(i1432_direct_write32_register(mn, (LONGSIZ32) cmdreg,
						E1432_SRC_XFRCMD_IDLE));

	/* indicate buffer is empty by setting WORDCOUNT_REG to 0 */
	SRC_CHECK(i1432_direct_write32_register(mn, (LONGSIZ32) xfrwdsreg, 0));
	break;

    case E1432_SRC_DATA_MODE_WAITA:
	/* is transfer done ? */
	/* xfrwds == 0 or A = Full */
	if (xfer_check_nowait(hw, chanID))
	    break;
	i1432_get_time(&startTime);
	do
	{
 	    i1432_get_time(&currentTime);
	    SRC_CHECK (i1432_direct_read32_register(mn, 
			(LONGSIZ32) xfrwdsreg, &xfrwds ));
	    SRC_CHECK (i1432_direct_read32_register(mn, 
			(LONGSIZ32) statusregA, &xfr_statusA));
#ifdef DPRINT
	    printf("xfr_waitA: mode (%d), xfrwds (%d), xfr_statusA (%x)\n",mode,xfrwds,xfr_statusA);
#endif
	    if (xfrwds != 0 && xfr_statusA != E1432_SRCBUF_FULL &&
		currentTime - startTime >= timeout_xfrwait)
		return i1432_la_print_error(mn->la, ERR1432_ACK_TIMEOUT);
	}
	while (xfrwds != 0 && xfr_statusA != E1432_SRCBUF_FULL);
	break;

    case E1432_SRC_DATA_MODE_WAITB:
	/* is transfer done ? */
	/* xfrwds == 0 or B = Full */
	if (xfer_check_nowait(hw, chanID))
	    break;
	i1432_get_time(&startTime);
	do
	{
 	    i1432_get_time(&currentTime);
	    SRC_CHECK (i1432_direct_read32_register(mn,
			(LONGSIZ32) xfrwdsreg, &xfrwds ));

	    SRC_CHECK (i1432_direct_read32_register(mn,
			(LONGSIZ32) statusregB, &xfr_statusB));
#ifdef DPRINT
	    printf("xfr_waitB: mode (%d), xfrwds (%d), xfr_statusB (%x)\n",mode,xfrwds,xfr_statusB);
#endif
	    if (xfrwds != 0 && xfr_statusB != E1432_SRCBUF_FULL &&
		currentTime - startTime >= timeout_xfrwait)
		return i1432_la_print_error(mn->la, ERR1432_ACK_TIMEOUT);
	}
	while (xfrwds != 0 && xfr_statusB != E1432_SRCBUF_FULL);
	break;

    case E1432_SRC_DATA_MODE_WAITAB:
	/* is transfer done ? */
	/* xfrwds == 0 or A = Full or B = Full */
	if(xfer_check_nowait(hw, chanID)) break;
	i1432_get_time(&startTime);
	do
	{
 	    i1432_get_time(&currentTime);
	    SRC_CHECK (i1432_direct_read32_register(mn, 
			(LONGSIZ32) xfrwdsreg, &xfrwds ));
	    SRC_CHECK (i1432_direct_read32_register(mn, 
			(LONGSIZ32) statusregA, &xfr_statusA));
	    SRC_CHECK (i1432_direct_read32_register(mn, 
			(LONGSIZ32) statusregB, &xfr_statusB));
#ifdef DPRINT
	    printf("xfr_waitA: mode (%d), xfrwds (%d), xfr_statusA (%x), xfr_statusB (%x)\n",mode,xfrwds,xfr_statusA,xfr_statusB);
#endif
	    if (xfrwds != 0 && (xfr_statusA != E1432_SRCBUF_FULL ||
				xfr_statusB != E1432_SRCBUF_FULL) &&
		currentTime - startTime >= timeout_xfrwait)
		return i1432_la_print_error(mn->la, ERR1432_ACK_TIMEOUT);
	}
	while (xfrwds != 0 && (xfr_statusA != E1432_SRCBUF_FULL ||
			       xfr_statusB != E1432_SRCBUF_FULL));
	break;

    case E1432_SRC_DATA_MODE_READID:
    case E1432_SRC_DATA_MODE_READNCLRSTATUS:
    case E1432_SRC_DATA_MODE_READSTATUS:
    case E1432_SRC_DATA_MODE_RAWREAD:
	/* is read done ? */
	/* xfrwds == -1 */
	i1432_get_time(&startTime);
	do
	{
	    i1432_get_time(&currentTime);
	    SRC_CHECK (i1432_direct_read32_register(mn, 
			(LONGSIZ32) xfrwdsreg, &xfrwds ));
#ifdef DPRINT
	    printf("xfr_READ: mode (%d), xfrwds (%d)\n",mode,xfrwds);
#endif

	    data_to_ptr = dataPtr;
	    data_from_ptr = (LONGSIZ32 * ) bufreg;
#ifdef DPRINT
	    printf("xfr_READ: data_to_ptr = %x, data_from_ptr = %x\n",data_to_ptr,data_from_ptr);
#endif

	    /*
	      SRC_CHECK (i1432_direct_read32_register(mn, (LONGSIZ32)data_from_ptr,
	      data_to_ptr ));
	    */
	    if (xfrwds == -1)
	    {
		for (i=0; i<numwords; i++)
		{
		    SRC_CHECK (i1432_direct_read32_register(mn, (LONGSIZ32)data_from_ptr, data_to_ptr ));
#ifdef DPRINT
		    printf("xfr_READ: i = %d; from %x ; to %x = %x\n",i,data_from_ptr,data_to_ptr,*data_to_ptr);
#endif
		    data_to_ptr++;
		    data_from_ptr++;
		}
	    }

	    if (xfrwds != -1 && currentTime - startTime >= timeout_xfrwait)
		return i1432_la_print_error(mn->la, ERR1432_ACK_TIMEOUT);
	}
	while (xfrwds != -1);

	SRC_CHECK (i1432_direct_write32_register(mn, (LONGSIZ32)cmdreg, (LONGSIZ32) E1432_SRC_XFRCMD_IDLE));
	/* indicate buffer is empty by setting WORDCOUNT_REG to 0 */
	/* effectively updating e1432_get_src_arbstates() */
	SRC_CHECK (i1432_direct_write32_register(mn, (LONGSIZ32)xfrwdsreg, (LONGSIZ32)0));
/*
	SRC_CHECK (i1432_direct_read32_register(mn, 
						(LONGSIZ32) xfrwdsreg, &xfrwds2 ));
	SRC_CHECK (i1432_direct_read32_register(mn, 
						(LONGSIZ32) cmdreg, &xfrcmd2 ));
	printf("xfr_READ: mode (%x), xfrwds (%d), xfrcmdreg (%x)\n\n",mode,xfrwds2,xfrcmd2);
*/
	break;
    }
    return 0;
}



/*
 *********************************************************************
 Returns:
       0 if no error.
      <0 if error.
 *********************************************************************
 */
SHORTSIZ16 EXPORT
e1432_get_src_arbstates(E1432ID hw, SHORTSIZ16  chanID,
        LONGSIZ32 *xfrbuf_wds, LONGSIZ32 *srcbuf_state_AB,
        LONGSIZ32 *srcbuf_state_A, LONGSIZ32 *srcbuf_state_B)
{
    E1432_MODULE_LIST_NODE *mn;
    SHORTSIZ16 error_src;
    LONGSIZ32 statusregA, statusregB, xfrwdsreg, cmdreg, bufreg;
    int     dram;
    
    TRACE_PRINTF(0, ("e1432_get_src_arbstates"
		     "(0x%p, %d, 0x%p, 0x%p, 0x%p, 0x%p)\n",
                     hw, chanID, xfrbuf_wds, srcbuf_state_AB,
		     srcbuf_state_A, srcbuf_state_B));

    /* check if real channel */
    SRC_CHECK (i1432_checkID(hw, chanID));
    if (chanID < 0)
	return i1432_print_error(ERR1432_EXPECT_CHANNEL);

    SRC_CHECK(i1432_get_module_from_chan(hw, chanID, &mn));

    SRC_CHECK(get_xfer_regs(hw, chanID, &xfrwdsreg, &statusregA,
			     &statusregB, &cmdreg, &bufreg, &dram));
    
    SRC_CHECK(i1432_direct_read32_register(mn, (LONGSIZ32) xfrwdsreg,
					   xfrbuf_wds));
    SRC_CHECK(i1432_direct_read32_register(mn, (LONGSIZ32) statusregA,
					   srcbuf_state_A));
    SRC_CHECK(i1432_direct_read32_register(mn, (LONGSIZ32) statusregB,
					   srcbuf_state_B));

    if (*srcbuf_state_A == *srcbuf_state_B)
	*srcbuf_state_AB = *srcbuf_state_A;
    else
	*srcbuf_state_AB = E1432_SRCBUF_RDY;

#ifdef DPRINT3
    printf("E1432_get_src_arbstates: chanID, *xfrbuf_wds, *srcbuf_state_AB, *srcbuf_state_A, *srcbuf_state_A\n	          (%x, %d, %d, %d, %d)\n", chanID, *xfrbuf_wds, *srcbuf_state_AB, *srcbuf_state_A, *srcbuf_state_A);
#endif
    return 0;
}


/*
 *********************************************************************
 Returns:
       1 if there is space in the substrate buffer for a srcbuffer_write
         data transfer, for the daughter board (or sca) of this channel,
	 while in arb mode.
       0 if no space.
      <0 if error.
 *********************************************************************
 */
SHORTSIZ16 EXPORT
e1432_check_src_arbrdy(E1432ID hw, SHORTSIZ16 chanID, SHORTSIZ16 mode)
{
    E1432_MODULE_LIST_NODE *mn;
    SHORTSIZ16 error_src;
    LONGSIZ32 xfr_statusA, xfr_statusB, statusregA, statusregB;
    LONGSIZ32 xfrwdsreg, cmdreg, bufreg, xfrwds;
    int     dram;
      
    TRACE_PRINTF(0, ("e1432_check_src_arbrdy(0x%p, %d, %d)\n",
                     hw, chanID, mode));

    /* check if real channel */
    SRC_CHECK (i1432_checkID(hw, chanID));
    if (chanID < 0)
	return i1432_print_error(ERR1432_EXPECT_CHANNEL);

    SRC_CHECK (i1432_get_module_from_chan(hw, chanID, &mn));

    SRC_CHECK (get_xfer_regs(hw, chanID, &xfrwdsreg, &statusregA,
			     &statusregB, &cmdreg, &bufreg, &dram));

    /*
      printf("e1432_check_src_arbrdy1: xfrwdsreg (%x), statusregA (%x),statusregB (%x), cmdreg (%x), bufreg (%x)\n",xfrwdsreg, statusregA, statusregB, cmdreg, bufreg);
    */
    
    SRC_CHECK (i1432_direct_read32_register(mn, (LONGSIZ32) xfrwdsreg,
					    &xfrwds ));
    SRC_CHECK (i1432_direct_read32_register(mn, (LONGSIZ32) statusregA,
					    &xfr_statusA));
    SRC_CHECK (i1432_direct_read32_register(mn, (LONGSIZ32) statusregB,
					    &xfr_statusB));

#ifdef DPRINT3
    printf("e1432_check_src_arbrdy: mode =  (%d),xfrwds (%d),xfr_statusA = (%d), xfr_statusB = (%d)\n",mode,xfrwds,xfr_statusA,xfr_statusB);
#endif

    /* is transfer buffer available ? */
    if (xfrwds != 0)
	return 0;

    switch (mode)
    {
    case E1432_SRC_DATA_MODE_RAWDATA:
    case E1432_SRC_DATA_MODE_RAWCMD:
	break;
    case E1432_SRC_DATA_MODE_A:
    case E1432_SRC_DATA_MODE_WAITA:
	if ((xfr_statusA & 0xff) == E1432_SRCBUF_FULL)
	    return 0;
	break;
    case E1432_SRC_DATA_MODE_B:
    case E1432_SRC_DATA_MODE_WAITB:
	if ((xfr_statusB & 0xff) == E1432_SRCBUF_FULL)
	    return 0;
	break;
    default:
	/* E1432_SRC_DATA_MODE_AB */
	/* E1432_SRC_DATA_MODE_WAITAB */
	if (((xfr_statusA & 0xff) == E1432_SRCBUF_FULL) &&
	    ((xfr_statusB & 0xff) == E1432_SRCBUF_FULL))
	    return 0;
	break;
    }
#ifdef DPRINT
    printf("e1432_check_src_arbrdy4: xfr_statusA = %d, xfr_statusB = %d\n",
	   xfr_statusA,xfr_statusB);
#endif
    return 1;
}

/*
 *********************************************************************
 Returns:
       1 if channel is shutdown at source start or while source is on
       0 if not shutdown, or after a reset meas.
      <0 if error.
 *********************************************************************
 */

SHORTSIZ16 EXPORT
e1432_check_src_shutdown(E1432ID hw, SHORTSIZ16 chanID)
{
    E1432_MODULE_LIST_NODE *mn;
    SHORTSIZ16 error, error_src, state;
    LONGSIZ32 status, mask;
    int sca, type;
    E1432_GROUP_LIST_NODE *gn;
    E1432_CHAN_LIST_NODE *cn;

    TRACE_PRINTF(0, ("e1432_check_src_shutdown(0x%p, %d)\n", hw, chanID));

    /* check for valid id */
    error = i1432_checkID(hw,chanID);
    if (error)
	return error;

    state = 0;

    /* check if channel or group */
    if (chanID < 0)
    {
	/* iterate thru group */
	gn = i1432_get_group_node(hw, chanID);
	cn = gn->chanlist;
	state = 0;
	while (cn)
	{
	    error = e1432_check_src_shutdown(hw, cn->chanID);
	    if (error < 0)
		return error;
	    if (error > 0)
		state = 1;

	    /* next channel */
	    cn = cn->next;
	}
    }
    else
    {
	type = (chanID & E1432_CHAN_TYPE_MASK) >> E1432_CHAN_TYPE_SHIFT;
	if (type == E1432_CHAN_TYPE_SOURCE)
	{
	    SRC_CHECK(i1432_get_module_from_chan(hw, chanID, &mn) );
	    sca = i1432_get_chan_sca(hw, chanID);
	    SRC_CHECK(i1432_direct_read32_register(mn,
						   E1432_SRC_STATUS_REG,
						   &status));
	    mask = E1432_SRC_STATUS_SHUTDOWN4;
	    if (sca == 0)
		mask = E1432_SRC_STATUS_SHUTDOWN0;
	    if (sca == 2)
		mask = E1432_SRC_STATUS_SHUTDOWN2;
	    if ((status & mask) != 0)
		state = 1; 

	    /* No clear on read here (unlike check_src_overread) */
	}
    }
    return state;
}

/*
 *********************************************************************
 Returns:
       1 if channel is overload at source start or while source is on
       0 if not overload, or after a reset meas.
      <0 if error.
 *********************************************************************
 */

SHORTSIZ16 EXPORT
e1432_check_src_overload(E1432ID hw, SHORTSIZ16 chanID)
{
    E1432_MODULE_LIST_NODE *mn;
    SHORTSIZ16 error, error_src, state;
    LONGSIZ32 status, mask;
    int sca, type;
    E1432_GROUP_LIST_NODE *gn;
    E1432_CHAN_LIST_NODE *cn;

    TRACE_PRINTF(0, ("e1432_check_src_overload(0x%p, %d)\n", hw, chanID));

    /* check for valid id */
    error = i1432_checkID(hw,chanID);
    if (error)
	return error;

    state = 0;

    /* check if channel or group */
    if (chanID < 0)
    {
	/* iterate thru group */
	gn = i1432_get_group_node(hw, chanID);
	cn = gn->chanlist;
	state = 0;
	while (cn)
	{
	    error = e1432_check_src_overload(hw, cn->chanID);
	    if (error < 0)
		return error;
	    if (error > 0)
		state = 1;

	    /* next channel */
	    cn = cn->next;
	}
    }
    else
    {
	type = (chanID & E1432_CHAN_TYPE_MASK) >> E1432_CHAN_TYPE_SHIFT;
	if (type == E1432_CHAN_TYPE_SOURCE)
	{
	    SRC_CHECK( i1432_get_module_from_chan(hw, chanID, &mn) );
	    sca = i1432_get_chan_sca(hw, chanID);
	    SRC_CHECK( i1432_direct_read32_register(mn,
						    E1432_SRC_STATUS_REG,
						    &status));
	    mask = E1432_SRC_STATUS_OVERLOAD4;
	    if (sca == 0)
		mask = E1432_SRC_STATUS_OVERLOAD0;
	    if (sca == 2)
		mask = E1432_SRC_STATUS_OVERLOAD2;
	    if ((status & mask) != 0)
		state = 1; 

	    /* No clear on read here (unlike check_src_overread) */
	}
    }
    return state;
}

/*
 *********************************************************************
 Returns:
       1 if channel arb buffer overread occures while source is on
       0 if no arb buffer overread, or after a reset meas.
      <0 if error.
 *********************************************************************
 */
SHORTSIZ16 EXPORT
e1432_check_src_overread(E1432ID hw, SHORTSIZ16 chanID)
{
    E1432_MODULE_LIST_NODE *mn;
    SHORTSIZ16 error, error_src, state;
    LONGSIZ32 status, mask;
    int sca, type;
    E1432_GROUP_LIST_NODE *gn;
    E1432_CHAN_LIST_NODE *cn;

    TRACE_PRINTF(0, ("e1432_check_src_overread(0x%p, %d)\n", hw, chanID));

    /* check for valid id */
    error = i1432_checkID(hw,chanID);
    if (error)
	return error;

    state = 0;

    /* check if channel or group */
    if (chanID < 0)
    {
	/* iterate thru group */
	gn = i1432_get_group_node(hw, chanID);
	cn = gn->chanlist;
	state = 0;
	while (cn)
	{
	    error = e1432_check_src_overread(hw, cn->chanID);
	    if (error < 0)
		return error;
	    if (error > 0)
		state = 1;

	    /* next channel */
	    cn = cn->next;
	}
    }
    else
    {
	type = (chanID & E1432_CHAN_TYPE_MASK) >> E1432_CHAN_TYPE_SHIFT;
	if (type == E1432_CHAN_TYPE_SOURCE)
	{
	    SRC_CHECK( i1432_get_module_from_chan(hw, chanID, &mn) );
	    sca = i1432_get_chan_sca(hw, chanID);
	    SRC_CHECK( i1432_direct_read32_register(mn,
						    E1432_SRC_STATUS_REG,
						    &status));
	    mask = E1432_SRC_STATUS_OVERREAD4;
	    if (sca == 0)
		mask = E1432_SRC_STATUS_OVERREAD0;
	    if (sca == 2)
		mask = E1432_SRC_STATUS_OVERREAD2;
	    if ((status & mask) != 0)
	    {
		state = 1;
		status &= ~mask; /* Clear bit that was read */
		SRC_CHECK(i1432_direct_write32_register(mn,
							E1432_SRC_STATUS_REG,
							status));
	    }
	}
    }
    return state;
}


/*
 *********************************************************************
 Returns:
       1 if channel dsp tx underrun (to dac) occurs while source is on
       0 if no dsp tx underrun, or after a reset meas.
      <0 if error.
 *********************************************************************
 */
SHORTSIZ16 EXPORT
e1432_check_src_underrun(E1432ID hw, SHORTSIZ16 chanID)
{
    E1432_MODULE_LIST_NODE *mn;
    SHORTSIZ16 error, error_src, state;
    LONGSIZ32 status, mask;
    int sca, type;
    E1432_GROUP_LIST_NODE *gn;
    E1432_CHAN_LIST_NODE *cn;

    TRACE_PRINTF(0, ("e1432_check_src_underrun(0x%p, %d)\n", hw, chanID));

    /* check for valid id */
    error = i1432_checkID(hw,chanID);
    if (error)
	return error;

    state = 0;

    /* check if channel or group */
    if (chanID < 0)
    {
	/* iterate thru group */
	gn = i1432_get_group_node(hw, chanID);
	cn = gn->chanlist;
	state = 0;
	while (cn)
	{
	    error = e1432_check_src_underrun(hw, cn->chanID);
	    if (error < 0)
		return error;
	    if (error > 0)
		state = 1;

	    /* next channel */
	    cn = cn->next;
	}
    }
    else
    {
	type = (chanID & E1432_CHAN_TYPE_MASK) >> E1432_CHAN_TYPE_SHIFT;
	if (type == E1432_CHAN_TYPE_SOURCE)
	{
	    SRC_CHECK( i1432_get_module_from_chan(hw, chanID, &mn) );
	    sca = i1432_get_chan_sca(hw, chanID);
	    SRC_CHECK( i1432_direct_read32_register(mn,
						    E1432_SRC_STATUS_REG,
						    &status));
	    mask = E1432_SRC_STATUS_UNDERRUN4;
	    if (sca == 0)
		mask = E1432_SRC_STATUS_UNDERRUN0;
	    if (sca == 2)
		mask = E1432_SRC_STATUS_UNDERRUN2;
	    if ((status & mask) != 0)
		state = 1; 

	    /* No clear on read here (unlike check_src_overread) */
	}
    }
    return state;
}

/********* source command vectors and commands *********/
#define SRCCMDV_EXTMSG		0xa2a2a2a2UL
#define SRCCMDV_CMDMODE		0xa3a3a3a3UL
#define SRCCMDV_SRCON		0xB1B1B1B1UL
#define SRCCMDV_ROMID		0x98989898UL
#define SRCCMDV_ROMDATE		0x9E9E9E9EUL
#define SRCCMDV_ERASESYS	0xA5A5A5A5UL
#define SRCCMDV_PROGSYS		0xA6A6A6A6UL
#define SRCCMDV_SRCBOOT		0x92929292UL

#define SRCCMD_BLOCKWRITE	0x0000000eUL
#define SRC_BLOCKWRITE_Y_SPACE	0x00000000UL
#define SRC_BLOCKWRITE_X_SPACE	0x00000001UL
#define SRC_BLOCKWRITE_P_SPACE	0x00000002UL
#define SRC_BLOCKWRITE_SIGADDR	0x00000068UL
#define SRC_BLOCKWRITE_XBUFADDR 0x00008000UL
#define SRC_BLOCKWRITE_ROMADDR  0x0000C000UL

#define SRCCMD_USERSTORE	0x0000000BUL
#define SRCCMD_USERCLR		0x0000000CUL
#define SRCCMD_USERCLR2		0x001D0001UL
#define SRCCMD_ERASESYS		0x001D0003UL
#define SRCCMD_PRGSYS		0x001E0001UL
#define SRCCMD_ACKMASK		0x00200000UL

#define	SRCCMD_BDPARMSRD	0x00000007UL
#define	SRCCMD_BDPARMSRESET	0x00000008UL

#define CHECKCODE	0x5c4d3e2fUL

/*************************************************************/

SHORTSIZ16 EXPORT
e1432_src_rxfr(E1432ID lhw, SHORTSIZ16 chanID, LONGSIZ32 numwords,
	       LONGSIZ32 *dptr, LONGSIZ32 mode)

{
    LONGSIZ32 i, words_left, xfrsize, numwordsmax;
    SHORTSIZ16 error_src;
    FLOATSIZ64 currentTime, startTime;
    numwordsmax = E1432_SRC_DATA_NUMWORDS_MAX;
#if 0
    LONGSIZ32 xfrbuf_wds, srcbuf_state_AB, srcbuf_state_A, srcbuf_state_B;
    SHORTSIZ16 error_src, error_src2;
#endif

/* test print */
#ifdef DPRINT3
    (void)printf("\ne1432_src_rxfr: mode %x, numwords %d\n",mode,numwords);
    (void)printf("buf[0-3]: %x, %x, %x, %x\n",dptr[0],dptr[1],dptr[2],dptr[3]);
#endif
    words_left = numwords;
    while (words_left > 0)
    {
	if (words_left <= numwordsmax)
	    xfrsize = words_left;
	else
	    xfrsize = numwordsmax;

	/* write  xfrsize words to the substrate */
/* test print*/
/*
	(void) printf("    e1432_write_srcbuffer_data() mode %x, xfrsize %d\n",mode,xfrsize);
*/
	error_src = e1432_write_srcbuffer_data(lhw, chanID, dptr,
					       (SHORTSIZ16) xfrsize,
					       (SHORTSIZ16)mode);
/*
	(void) printf("    e1432_write_srcbuffer_data returns: %d\n",error_src);
*/

	if (error_src == ERR1432_ACK_TIMEOUT)
	{
	    (void) printf("e1432_src_rxfr: e1432_write_srcbuffer_data - ERR1432_ACK_TIMEOUT \n");
/*
	    error_src2 = e1432_get_src_arbstates(lhw, chanID, &xfrbuf_wds,
						 &srcbuf_state_AB, &srcbuf_state_A,
						 &srcbuf_state_B);
*/    
	    return error_src;
	}
	if (error_src != 0)
	{
	    (void)i1432_get_time(&startTime);
	    i = 1;
	    (void)i1432_get_time(&currentTime);
	    while ( ( currentTime - startTime < 20) && (error_src != 0) )
	    {
		i=i+1;
/*
		(void) printf("    e1432_write_srcbuffer_data, try = %d\n",i);
*/
		error_src = e1432_write_srcbuffer_data(lhw, chanID,
					       dptr,
					       (SHORTSIZ16) xfrsize,
					       (SHORTSIZ16)mode);
/*
                (void) printf("    e1432_write_srcbuffer_data returns: %d\n",error_src);
*/
		if (error_src == ERR1432_ACK_TIMEOUT)
		{
		    (void) printf("e1432_src_rxfr: e1432_write_srcbuffer_data - ERR1432_ACK_TIMEOUT \n");
		    return error_src;
		}
		(void)i1432_get_time(&currentTime);
	    }	
	    if ((currentTime-startTime >= 20) || (error_src != 0))
	    {
		(void) printf("e1432_src_rxfr: transfer failed error = %d\n",error_src);	
		return error_src;
	    }
	}
	dptr = dptr + xfrsize;
	words_left = words_left - xfrsize;
    }
    return 0;
}

/***************** e1432_src_get_fwrev ************************/
SHORTSIZ16 EXPORT
e1432_src_get_fwrev(LONGSIZ32 *dptr, LONGSIZ32 *fwid, LONGSIZ32 *fwdate, LONGSIZ32 numwords)
{
    LONGSIZ32 size, headersize, checkcode, romsize, trailer;
    LONGSIZ32 *dptr_tmp;
/*
 *      Source rom input file, srcrom.bin, is created by  srclod2rom.c
 *	It consist of:
 *      	Word is 32 bits, 4 bytes
 *		Header[6]
 *		  (totalsize, headersize, checkcode, id, id2,romsize)
 *                MSB first
 *		ROM image [] MSB first
 *      		MSB is $00, followed by 3 bytes value
 *		Trailer[1] (checkcode) MSB first
 */

    dptr_tmp = dptr;

    size = *dptr_tmp++;
    headersize = *dptr_tmp++;
    checkcode = *dptr_tmp++;
    * fwid = *dptr_tmp++;
    * fwdate = *dptr_tmp++;
    romsize = *dptr_tmp++;
    * fwdate = * fwdate & 0x00FFFFFF;
/* to avoid year 2000 problem */
    if (* fwdate < 0x00950000) * fwdate = * fwdate | 0x20000000;
    else * fwdate = * fwdate | 0x19000000;

/*
    (void)printf("e1432_src_get_fwrev: total size(words): %d, headersize: %d, checkcode: %x\n",size,headersize,checkcode);
    (void)printf("e1432_src_get_fwrev: fwid: %x, fwdate: %x, rom size(words): %d\n",*fwid, *fwdate,romsize);
    (void)printf("e1432_src_get_fwrev: file begins at %x, rom begins at: %x\n",dptr,dptr_tmp);
*/

/* check for valid source rom code */

/* if checkcode is not correct could be byte swap */
    if (checkcode != CHECKCODE) return ERR1432_SRC_ROMFILE_CHECKCODE;

/*
if file size does not match size in header, file is incomplete or wrong
type
*/
    if (size != numwords) return ERR1432_SRC_ROMFILE_SIZE;

/* if last word not = to check word, file is incomplete or wrong type */

    dptr_tmp = dptr;
    dptr_tmp = dptr_tmp+headersize+romsize;
    trailer = *dptr_tmp;
/*
    (void)printf("e1432_src_get_fwrev: trailer is %x, at %x\n",trailer,dptr_tmp);
*/
    if (trailer != CHECKCODE) return ERR1432_SRC_ROMFILE_TRAILER;

    return 0;
}


/***************** e1432_src_prog_romimage ************************/
SHORTSIZ16 EXPORT
e1432_src_prog_romimage(E1432ID hw,SHORTSIZ16 src_chan, LONGSIZ32 *dptr, LONGSIZ32 numwords)
{
    int error_src =0;
    LONGSIZ32 bufr[256];
    LONGSIZ32 xfrbufr[4096];
    LONGSIZ32 *dptr_tmp;
    /*LONGSIZ32 *dptr_old;*/
    LONGSIZ32 *xfrptr;
    /*LONGSIZ32 fwid;*/
    /*LONGSIZ32 fwdate;*/
    LONGSIZ32 size, headersize, checkcode, romsize, trailer;
    LONGSIZ32 i, xfrsize, romaddr,wordsleft, block;
    FLOATSIZ64 startTime,currentTime;
    
/*
******** check source rom file array ********
*/
    dptr_tmp = dptr;

    size = *dptr_tmp++;
    headersize = *dptr_tmp++;
    checkcode = *dptr_tmp++;
    /*fwid = **/dptr_tmp++;
    /*fwdate = **/dptr_tmp++;
    romsize = *dptr_tmp++;

    /*
    (void)printf("e1432_src_get_fwrev: total size(words): %d, headersize: %d, checkcode: %x\n",size,headersize,checkcode);
    (void)printf("e1432_src_get_fwrev: fwid: %x, fwdate: %x, rom size(words): %d\n",*fwid, *fwdate,romsize);
    (void)printf("e1432_src_get_fwrev: file begins at %x, rom begins at: %x\n",dptr,dptr_tmp);
    */

    /* check for valid source rom code */

    /* if checkcode is not correct could be byte swap */
    if (checkcode != CHECKCODE) return ERR1432_SRC_ROMFILE_CHECKCODE;

    /*
      if file size does not match size in header, file is incomplete or wrong
      type
      */
    if (size != numwords) return ERR1432_SRC_ROMFILE_SIZE;

    /* if last word not = to check word, file is incomplete or wrong type */

    dptr_tmp = dptr;
    dptr_tmp = dptr_tmp+headersize+romsize;
    trailer = *dptr_tmp;
    /*
    (void)printf("e1432_src_get_fwrev: trailer is %x, at %x\n",trailer,dptr_tmp);
    */
    if (trailer != CHECKCODE) return ERR1432_SRC_ROMFILE_TRAILER;

/*
******** check source channel ********
*/
    SRC_CHECK (i1432_checkID(hw, src_chan));
    if((src_chan < 0x1000)|| (src_chan > 0x1FFF)) return ERR1432_NO_CHANNEL;

/*
******** Initialize transfer buffer for the source channel ********
*/

/*
    SRC_CHECK(e1432_set_active(hw, src_chan, E1432_CHANNEL_ON));
    SRC_CHECK(e1432_set_srcbuffer_init(hw, src_chan,E1432_SRCBUFFER_INIT_XFER));
*/

/*
******** erase system area ********
*/
    (void)printf("\n**** erasing system area of source board flash rom, do not interrupt.\n");

/* Timeout error if E1432_SRC_DATA_MODE_RAWCMD (which waits for ack) is
   used.  Therefore using _RAWNACKCMD which does not wait, then check
   status with nonclearing status read, _READNCLRSTATUS, and ending
   by clearing status with _READSTATUS */
 
    /* clear status */
    SRC_CHECK(e1432_src_rxfr(hw, src_chan, 1, bufr,E1432_SRC_DATA_MODE_READSTATUS));
	

    bufr[0] = SRCCMD_ERASESYS<<8;
    SRC_CHECK(e1432_src_rxfr(hw, src_chan, 1, bufr,
			       E1432_SRC_DATA_MODE_RAWDATA));
    bufr[0] = SRCCMDV_ERASESYS;
    (void)i1432_get_time(&startTime);
    currentTime= startTime;
    SRC_CHECK(e1432_src_rxfr(hw, src_chan, 1, &bufr[0],
			       E1432_SRC_DATA_MODE_RAWNACKCMD));

    /* wait for completion */
    error_src =0;
    bufr[0] = 0;
    SRC_CHECK(e1432_src_rxfr(hw, src_chan, 1, &bufr[0],
				   E1432_SRC_DATA_MODE_READNCLRSTATUS));
/*
    (void)i1432_get_time(&currentTime);
    (void)printf("Flash ROM erase time = %f seconds\n",currentTime-startTime);
    (void)printf("source SISR reg: %x\n",bufr[0]);
*/
    while(((bufr[0] & SRCCMD_ACKMASK) == 0) && (currentTime-startTime < 60))
    {
	bufr[0] = 0;
	SRC_CHECK(e1432_src_rxfr(hw, src_chan, 1, bufr,
				   E1432_SRC_DATA_MODE_READNCLRSTATUS));
	(void)i1432_get_time(&currentTime);
/*
	(void)printf("source SISR reg: %x\n",bufr[0]);
	(void)i1432_get_time(&currentTime);
	(void)printf("Flash ROM erase time = %f seconds\n",currentTime-startTime);
*/
    }
    (void)printf("source board flash rom erased, time = %f seconds\n",currentTime-startTime);
    /* clear status */
    SRC_CHECK(e1432_src_rxfr(hw, src_chan, 1, bufr,
			       E1432_SRC_DATA_MODE_READSTATUS));
/*
    (void)printf("source SISR reg: %x\n",bufr[0]);
*/

/*
******** programming stuff goes here ********
*/

    /* set source in cmdmode */
    bufr[0] = SRCCMDV_CMDMODE;
    SRC_CHECK(e1432_src_rxfr(hw, src_chan, 1, bufr,
			       E1432_SRC_DATA_MODE_RAWCMD));
    /* determine pointer to data */
    dptr_tmp = dptr+headersize;

    /* determine rom address */
    romaddr = SRC_BLOCKWRITE_ROMADDR;
    wordsleft = romsize;
/*
    (void)printf("source rom size 0x%x words; 0x%x bytes\n",romsize,3 * romsize);
    (void)printf("source rom size %d words; %d bytes\n",romsize,3 * romsize);
*/

    block=1;
    
/* for numwords of program */
    while(wordsleft>0)
    {
	
	/* determine xfer blocksize */
	xfrsize = 4096;
	if (wordsleft < 4096)
	{
	    xfrsize = wordsleft;
	}
	wordsleft = wordsleft -  xfrsize;

    
	/* setup block write to x memory */
	xfrptr = &xfrbufr[0];
	/*dptr_old=dptr_tmp;*/
	
	/* shift words left since only top 3 bytes sent to 56000 *********
	   dont modify file array, just shift the block that is transferred
	   */

	for (i=0; i<xfrsize; i++) xfrbufr[i]= *dptr_tmp++ << 8;

/* write upto 4k to x memory 0x8000 .. 0x9000  via transfer buffer */
/*
******** send extended message (data and cmd) to set up program transfer ********
*/
/*
	(void)printf("\n***source rom transfering block %d of %d words, (0x%x bytes)\n",block,xfrsize,3 * xfrsize);
	(void)printf("      from 0x%x to 0x%x\n",dptr_old,dptr_tmp-1);
*/
	bufr[0] = SRCCMD_BLOCKWRITE<<8;
	bufr[1] = SRC_BLOCKWRITE_X_SPACE<<8;
	bufr[2] = SRC_BLOCKWRITE_XBUFADDR<<8;
	SRC_CHECK(e1432_src_rxfr(hw, src_chan, 3, bufr,
				   E1432_SRC_DATA_MODE_RAWDATA));
	bufr[0] = SRCCMDV_EXTMSG;
	SRC_CHECK(e1432_src_rxfr(hw, src_chan, 1, bufr,
				   E1432_SRC_DATA_MODE_RAWCMD));
    
/*
******** transfer data ********
*/
	SRC_CHECK(e1432_src_rxfr(hw, src_chan,xfrsize,xfrptr,E1432_SRC_DATA_MODE_RAWDATA));


/*
******** end transfer by placing hostport in command buffer mode  ********
*/
	bufr[0] = SRCCMDV_CMDMODE;
	SRC_CHECK(e1432_src_rxfr(hw, src_chan, 1, bufr,
				   E1432_SRC_DATA_MODE_RAWCMD));
/*
******** program this block to flash ROM  *******************
*/
/*
	(void)printf("\n**** programming rom at %x\n",romaddr);
*/
	(void)printf("**** programming source board rom block %d, do not interrupt.\n",block);

/* Timeout error if E1432_SRC_DATA_MODE_RAWCMD (which waits for ack) is
   used.  Therefore using _RAWNACKCMD which does not wait, then check
   status with nonclearing status read, _READNCLRSTATUS, and ending
   by clearing status with _READSTATUS
*/
	/* clear status */
	SRC_CHECK(e1432_src_rxfr(hw, src_chan, 1, bufr,E1432_SRC_DATA_MODE_READSTATUS));
	
	bufr[0] = SRCCMD_PRGSYS<<8;
	bufr[1] = SRC_BLOCKWRITE_X_SPACE<<8;
	bufr[2] = SRC_BLOCKWRITE_XBUFADDR<<8;
	bufr[3] = romaddr<<8;
	bufr[4] = xfrsize<<8;
	SRC_CHECK(e1432_src_rxfr(hw, src_chan, 5, bufr,
				   E1432_SRC_DATA_MODE_RAWDATA));
	bufr[0] = SRCCMDV_PROGSYS;
	(void)i1432_get_time(&startTime);
	currentTime= startTime;
	SRC_CHECK(e1432_src_rxfr(hw, src_chan, 1, &bufr[0],
				   E1432_SRC_DATA_MODE_RAWNACKCMD));

	/* wait for completion */
	error_src =0;
	bufr[0] = 0;
	SRC_CHECK(e1432_src_rxfr(hw, src_chan, 1, &bufr[0],
				   E1432_SRC_DATA_MODE_READNCLRSTATUS));
/*
	(void)i1432_get_time(&currentTime);
	(void)printf("Flash ROM programming time = %f seconds\n",currentTime-startTime);
	(void)printf("source SISR reg: %x\n",bufr[0]);
*/

	while(((bufr[0] & SRCCMD_ACKMASK) == 0) && (currentTime-startTime < 60))
	{
	    bufr[0] = 0;
	    SRC_CHECK(e1432_src_rxfr(hw, src_chan, 1, bufr,
				       E1432_SRC_DATA_MODE_READNCLRSTATUS));
	    (void)i1432_get_time(&currentTime);
/*
	    (void)printf("source SISR reg: %x\n",bufr[0]);
	    (void)i1432_get_time(&currentTime);
	    (void)printf("Flash ROM programming time = %f seconds\n",currentTime-startTime);
*/
	}
	(void)printf("source board rom block %d programmed, time = %f seconds\n",block,currentTime-startTime);
	    /* clear status */
	SRC_CHECK(e1432_src_rxfr(hw, src_chan, 1, bufr,E1432_SRC_DATA_MODE_READSTATUS));
/*
	(void)printf("source SISR reg: %x\n",bufr[0]);
*/
	for( i=0;i<(xfrsize*3);i++) 
	{
	    romaddr = romaddr+1;
	    if (romaddr == 0x10000) romaddr = 0x18000;
	    if (romaddr == 0x20000) romaddr = 0x28000;
	    if (romaddr == 0x30000) romaddr = 0x38000;
	}
	block = block+1;
    }	
/*
    (void)printf("Flash ROM PROGRAMMED, last byte = %x\n",romaddr-1);
*/
    /* reboot source */
    (void)printf("\nsource board rom programming completed\n");
    (void)printf("rebooting source\n");
    bufr[0] = SRCCMDV_SRCBOOT;
    SRC_CHECK(e1432_src_rxfr(hw, src_chan, 1, &bufr[0],
				   E1432_SRC_DATA_MODE_RAWCMD));
    
    return 0;
}


/***************** e1432_src_get_rev ************************/
SHORTSIZ16 EXPORT
e1432_src_get_rev(E1432ID hw,SHORTSIZ16 src_chan, LONGSIZ32 *romid,
		  LONGSIZ32 *romdate, LONGSIZ32 *bdid, char *bddate)
{
    int error_src =0;
    LONGSIZ32 bufr[256];
    LONGSIZ32 bufrindx;
    char *bddatetmp;
    
/* bdparms set at manufacture */
struct bdpstruct1 {
  ULONGSIZ32 structrevid; /* revid of this structure (0x000001) */
  ULONGSIZ32 hwrevid;     /* revid of hardware (0x000001) */
  ULONGSIZ32 date[12];    /* date last changed  1char per word */
  ULONGSIZ32 ch1ghb;      /* ch1 hi band gain correction, 10/13.92 (0x5bf43b)*/
  ULONGSIZ32 ch1glb;      /* ch1 low band gain correction, 10/13.92 (0x5bf43b) */
  ULONGSIZ32 ch1gcalbus;  /* ch1 calout bus gain correction, 2/3 (0x55566d) */
  ULONGSIZ32 ch1gcalout;  /* ch1 cal output gain correction, 2/3 (0x55566d) */
  ULONGSIZ32 ch1oshb_0;   /* ch1 dc offset dac0 zero hi band correction */
  ULONGSIZ32 ch1oshb_1;   /* ch1 dc offset dac1 zero hi band correction */
  ULONGSIZ32 ch1oslb_0;   /* ch1 dc offset dac0 zero lo band correction */
  ULONGSIZ32 ch1oslb_1;   /* ch1 dc offset dac1 zero lo band correction */
  ULONGSIZ32 ch2ghb;      /* ch2 hi band gain correction, 10/13.92 (0x5bf43b) */
  ULONGSIZ32 ch2glb;      /* ch2 low band gain correction, 10/13.92 (0x5bf43b) */
  ULONGSIZ32 ch2gcalbus;  /* ch2 calout bus gain correction, 2/3 (0x55566d) */
  ULONGSIZ32 ch2gcalout;  /* ch2 cal output gain correction, 2/3 (0x55566d) */
  ULONGSIZ32 ch2oshb_0;   /* ch2 dc offset dac0 zero hi band correction */
  ULONGSIZ32 ch2oshb_1;   /* ch2 dc offset dac1 zero hi band correction */
  ULONGSIZ32 ch2oslb_0;   /* ch2 dc offset dac0 zero lo band correction */
  ULONGSIZ32 ch2oslb_1;   /* ch2 dc offset dac1 zero lo band correction */
  ULONGSIZ32 unused1[60-30];      /* unused to fill first 60 words */
  ULONGSIZ32 hwrev_1[16];     /* hardware revision character string 1char per word */
  ULONGSIZ32 comments[44];    /* comments string  1char per word */
} bdparms1;

    ULONGSIZ32 *bdparms1ptr;

    bddatetmp=bddate;
    
/*
******** check source channel ********
*/
    SRC_CHECK (i1432_checkID(hw, src_chan));
    if((src_chan < 0x1000)|| (src_chan > 0x1FFF)) return ERR1432_NO_CHANNEL;


/*
******** Initialize transfer buffer for the source channel ********
*/

/*
    SRC_CHECK(e1432_set_active(hw, src_chan, E1432_CHANNEL_ON));
    SRC_CHECK(e1432_set_srcbuffer_init(hw, src_chan,E1432_SRCBUFFER_INIT_XFER));
*/

/*
******** read rom id and date ********
*/
    bufr[0] = SRCCMDV_CMDMODE;
    SRC_CHECK(e1432_src_rxfr(hw, src_chan, 1, bufr,
			 E1432_SRC_DATA_MODE_RAWCMD));

    bufr[0] = SRCCMDV_ROMID;
    SRC_CHECK(e1432_src_rxfr(hw, src_chan, 1, bufr,
			 E1432_SRC_DATA_MODE_RAWCMD));

    bufr[0] = 0;
    bufr[1] = 0;
    SRC_CHECK(e1432_src_rxfr(hw, src_chan, 2, bufr,
			 E1432_SRC_DATA_MODE_RAWREAD));
    *romid = bufr[1]>>8 & 0x00FFFFFF;
/*
    (void)printf("source ROM ID:  %x hex, %d dec\n",*romid,*romid);
*/

    bufr[0] = SRCCMDV_ROMDATE;
    SRC_CHECK(e1432_src_rxfr(hw, src_chan, 1, bufr,
			 E1432_SRC_DATA_MODE_RAWCMD));

    bufr[0] = 0;
    bufr[1] = 0;
    SRC_CHECK(e1432_src_rxfr(hw, src_chan, 2, bufr,
			 E1432_SRC_DATA_MODE_RAWREAD));
    *romdate = bufr[1]>>8 & 0x00FFFFFF;
/* to avoid year 2000 problem */
    if (*romdate < 0x00950000) *romdate = *romdate | 0x20000000;
    else *romdate = *romdate | 0x19000000;

/*
    (void)printf("source ROM date: %x\n",*romdate);
*/

/*
******** read board parameters set during manufacture ********
*/
    bufr[0] = SRCCMDV_CMDMODE;
    SRC_CHECK(e1432_src_rxfr(hw, src_chan, 1, bufr,
			 E1432_SRC_DATA_MODE_RAWCMD));
    /* must read in 2 blocks of 60 (+1 garbage) words due to CMDMODE
       dsp host port data buffer 64 wd limits */
    bdparms1ptr = (ULONGSIZ32 *) &bdparms1;

    /* read bd parms block 1 */
    bufr[0] = SRCCMD_BDPARMSRD<<8;
    bufr[1] = 1<<8;
    SRC_CHECK(e1432_src_rxfr(hw, src_chan, 2, bufr,
			 E1432_SRC_DATA_MODE_RAWDATA));

    bufr[0] = SRCCMDV_EXTMSG;
    SRC_CHECK(e1432_src_rxfr(hw, src_chan, 1, bufr,
			 E1432_SRC_DATA_MODE_RAWCMD));

    bufr[0] = 0;
    bufr[1] = 0;
    SRC_CHECK(e1432_src_rxfr(hw, src_chan, BDPARM_SIZE + 1, bufr,
			 E1432_SRC_DATA_MODE_RAWREAD));

    for (bufrindx = 0; bufrindx < BDPARM_SIZE; bufrindx++)
    { 
/*
    (void)printf("    bufr [%d] = %x, %.4c\n",bufrindx+1,bufr[bufrindx+1],bufr[bufrindx+1]);
*/
	/* skip first word as it is garbage */
	*bdparms1ptr++ = bufr[bufrindx+1]>>8 & 0x00FFFFFF;
    }

    /* read bd parms block 2 */
    bufr[0] = SRCCMD_BDPARMSRD<<8;
    bufr[1] = 2<<8;
    SRC_CHECK(e1432_src_rxfr(hw, src_chan, 2, bufr,
			 E1432_SRC_DATA_MODE_RAWDATA));

    bufr[0] = SRCCMDV_EXTMSG;
    SRC_CHECK(e1432_src_rxfr(hw, src_chan, 1, bufr,
			 E1432_SRC_DATA_MODE_RAWCMD));

    bufr[0] = 0;
    bufr[1] = 0;
    SRC_CHECK(e1432_src_rxfr(hw, src_chan, BDPARM_SIZE + 1, bufr,
			 E1432_SRC_DATA_MODE_RAWREAD));

    for (bufrindx = 0; bufrindx < BDPARM_SIZE; bufrindx++)
    { 
/*
        (void)printf("    bufr [%d] = %x, %.4c\n",bufrindx+1,bufr[bufrindx+1],bufr[bufrindx+1]);
*/
	/* skip first word as it is garbage */
	*bdparms1ptr++ = bufr[bufrindx+1]>>8 & 0x00FFFFFF;
    }


    bdparms1ptr = (ULONGSIZ32 *) &bdparms1;
    bdparms1ptr++;

/*
    (void)printf("\nsource board parameters for source = %x\n",src_chan);
    (void)printf("    revid of hardware: %x hex, %d dec\n",*bdparms1ptr,*bdparms1ptr);
*/
    *bdid = *bdparms1ptr;
    /* put 12 words into string */
    for (bufrindx = 0; bufrindx < 12; bufrindx++)
    {
	/* Extract fourth byte of four-byte value. */
#if SWAP
	*bddatetmp++ =  (char)(*bdparms1ptr);
#else
	*bddatetmp++ = *(((char *) bdparms1ptr) + 3);
#endif
	bdparms1ptr++;
    }
    *bddatetmp = 0x0;


#ifdef PRINTBDPARMS

	    bdparms1ptr = (ULONGSIZ32 *) &bdparms1;

	    (void)printf("\nsource board parameters for source = %x\n",src_chan);
	    (void)printf("    revid of this structure: %x\n",*bdparms1ptr++);
	    (void)printf("    revid of hardware: %x hex, %d dec\n",*bdparms1ptr,*bdparms1ptr);
	    (void)printf("    date last changed char [12]: %s\n",bddate);
	    bdparms1ptr = bdparms1ptr+13;
	    (void)printf("    ch1 hi band gain correction: %x\n",*bdparms1ptr++);
	    (void)printf("    ch1 low band gain correction: %x\n",*bdparms1ptr++);
	    (void)printf("    ch1 calout bus gain correction: %x\n",*bdparms1ptr++);
	    (void)printf("    ch1 cal output gain correction: %x\n",*bdparms1ptr++);
	    (void)printf("    ch1 dc offset dac0 zero hi band correction: %x\n",*bdparms1ptr++);
	    (void)printf("    ch1 dc offset dac1 zero hi band correction: %x\n",*bdparms1ptr++); 
	    (void)printf("    ch1 dc offset dac0 zero lo band correction: %x\n",*bdparms1ptr++); 
	    (void)printf("    ch1 dc offset dac1 zero lo band correction: %x\n",*bdparms1ptr++); 
	    (void)printf("    ch2 hi band gain correction: %x\n",*bdparms1ptr++);
	    (void)printf("    ch2 low band gain correction: %x\n",*bdparms1ptr++);
	    (void)printf("    ch2 calout bus gain correction: %x\n",*bdparms1ptr++);
	    (void)printf("    ch2 cal output gain correction: %x\n",*bdparms1ptr++);
	    (void)printf("    ch2 dc offset dac0 zero hi band correction: %x\n",*bdparms1ptr++);
	    (void)printf("    ch2 dc offset dac1 zero hi band correction: %x\n",*bdparms1ptr++);
	    (void)printf("    ch2 dc offset dac0 zero lo band correction: %x\n",*bdparms1ptr++);
	    (void)printf("    ch2 dc offset dac1 zero lo band correction: %x\n",*bdparms1ptr++);
#endif

    return error_src;
}
